; Dll.asm - Minimal DLL
;
; How to assemble this to a DLL:
;
; Making a DLL is really only a little bit more involved than making a
; standard executable. Once you've got your source file, assemble it in
; the normal fashion to an object file.
;
; Now, before you link it, you need what's called a 'definitions' file. A
; definitions file simply tells the linker what symbols (labels declared
; as GLOBAL in NASM) you want other people to be able to see. This is
; because not only will the linker create a .DLL file, but also a .LIB
; file which you'll need when you want to link your DLL to another
; program. A definitions file looks like:
;
; LIBRARY Dll
; EXPORTS
;     _Function1@0
;     _Function2@0
;
; All this says is that the library's called Dll and that there are two
; symbols available - Function1@0 and Function2@0. This is just like when
; in User32.dll there are lots of symbols available, eg _MessageBoxA@16.
; The actual symbol names themselves can be anything though.
;
; After you've created this (for this example, there's one already done,
; called Dll.def), you can link it via:
;
; link /entry:DllMain /subsystem:windows /dll /def:Dll.def /libpath:<libpath> Dll.obj
;
; This will create three files - Dll.dll, Dll.lib and Dll.exp. When you want
; to create an application which uses these functions, you'll need Dll.lib,
; but I'll detail this in Dll_test.asm, which shows how to call the functions
; in the DLL produced by this file.
;

%define _WINMESSAGES_
%include "Gaz\Win32\Include\Windows.inc"

[BITS 32]
[section .text]

procglobal DllMain, hinstDLL, fdwReason, lpvReserved
	;
	; DLLs work differently to standard applications, but still have an
	; entry point, traditionally called DllMain. This part of the DLL
	; should set up anything for the other DLL functions, for example
	; allocating memory, etc.
	;
	; At the start of the DLL, we call the special macro LibMainPrologue
	; which preserves registers, etc.
	;
	LibMainPrologue
	;
	; When a DLL is called, fdwReason holds one of four values, which
	; indicate why the DLL is being called:
	;
	; DLL_PROCESS_ATTACH - the DLL has been loaded into the address space
	;   of a process that's either just started or has called
	;   LoadLibrary to load the DLL in manually. If you get this value,
	;   you should perform any initialisation that you need for the DLL's
	;   functions, for example memory allocation.
	;
	; DLL_THREAD_ATTACH - the process the DLL is attached to has created
	;   a new thread.
	;
	; DLL_PROCESS_DETACH - the DLL is being unloaded from the address
	;   space of the process, either it's exiting cleanly or is being
	;   unloaded via FreeLibrary. Either way, you should free up any
	;   allocated memory, etc.
	;
	; DLL_THREAD_DETACH - a thread is exiting cleanly.
	;
	; We don't actually have any initialisation to do here, so this bit
	; is blank in this simple example
	;
	; At the end of DllMain, we can set a return value (in eax). It's only
	; used if the DLL was called with fdwReason holding DLL_PROCESS_ATTACH,
	; with the other values it's ignored. The return value is used to tell
	; the process whether the DLL has initialised successfully or not. A
	; value of zero indicates no, whilst a non-zero value indicates yes.
	; If the process is starting up, and a zero value is returned, the
	; application will fail with a dialog box stating 'A required .DLL
	; file, Name.DLL is missing' or something like that and fail to start.
	; If it was called via LoadLibrary, the process can trap the error and
	; decide whether it's fatal or not.
	;
	; flag the DLL initialised successfully
	;
	mov	eax, 1
	;
	; at the end of the DllMain, we call the special macro LibMainEpilogue
	; to tidy registers, etc.
	;
	LibMainEpilogue
endproc
;
;-----------------------------------------------------------------------
;
; Here's where we define our DLL functions, and we use the procglobal
; macro to make a callable function just like we do with WinMain.
; A DLL function is just a piece of code that can be called and ends
; with a ret instruction. And that's all it is.
;
; Well, you need to define a register convention and stick to it, but
; it makes sense to just do the same as for Win32 API calls.
;
; The first function simply displays a message box and takes no paraemters.
;
procglobal DllTest
	;
	; Invoke the messagebox
	;
	TEXTlocal _msgbox_title, 'Minimal DLL',0
	TEXTlocal _msgbox_text, 'Hello from the minimal DLL!',0
	sc MessageBox, NULL, ._msgbox_text, ._msgbox_title, MB_OK
endproc
;
; The second function demonstrates a function that takes two parameters:
;
;   1. Address of messagebox title
;   2. Address of messagebox text
;
; It uses the passed addresses when it displays a messagebox.
;
; Notice how a DLL is mapped into the same address space as the process
; so the DLL has access to all the memory the process has, and also notice
; how we can simply pass addresses between the process and the DLL and not
; worry about segment registers, etc. as Windows handles the correct mapping
; of the DLL into the process' address space.
;
procglobal DllTest2, _msgbox_title, _msgbox_text
	;
	; invoke the messagebox
	;
	sc MessageBox, NULL, ._msgbox_text, ._msgbox_title, MB_OK
endproc

[section .data]
